// Language Courses Website - Version 3 (HTML/CSS/JS)

// Languages data
const languages = [
  { 
    id: 'english', 
    name: 'Английский', 
    icon: 'icons/course-globe.svg', 
    color: '#3B82F6',
    description: 'Самый популярный язык в мире'
  },
  { 
    id: 'spanish', 
    name: 'Испанский', 
    icon: 'icons/course-languages.svg', 
    color: '#EF4444',
    description: 'Язык страсти и культуры'
  },
  { 
    id: 'chinese', 
    name: 'Китайский', 
    icon: 'icons/course-book.svg', 
    color: '#F59E0B',
    description: 'Откройте двери в Азию'
  },
  { 
    id: 'german', 
    name: 'Немецкий', 
    icon: 'icons/course-award.svg', 
    color: '#10B981',
    description: 'Язык бизнеса и науки'
  }
]

// Initialize page
function initPage() {
  const currentPage = window.location.pathname.includes('enroll.html') ? 'enroll' : 'courses'
  
  if (currentPage === 'enroll') {
    initEnrollPage()
  } else {
    initCoursesPage()
  }
  
  initSlider()
}

// Initialize courses page
function initCoursesPage() {
  loadCourses()
  updateStats()
  
  // Poll for storage changes
  setInterval(() => {
    loadCourses()
    updateStats()
  }, 500)
}

// Load courses from localStorage
function loadCourses() {
  const saved = localStorage.getItem('languageCourses')
  let courses = []
  
  if (saved) {
    try {
      courses = JSON.parse(saved)
    } catch (e) {
      courses = []
    }
  } else {
    // Default courses
    courses = [
      {
        id: 1,
        language: 'Английский',
        level: 'Intermediate',
        progress: 65,
        rating: 5,
        icon: 'icons/course-globe.svg'
      },
      {
        id: 2,
        language: 'Испанский',
        level: 'Beginner',
        progress: 30,
        rating: 4,
        icon: 'icons/course-languages.svg'
      },
      {
        id: 3,
        language: 'Китайский',
        level: 'Advanced',
        progress: 85,
        rating: 5,
        icon: 'icons/course-book.svg'
      }
    ]
    localStorage.setItem('languageCourses', JSON.stringify(courses))
  }
  
  return courses
}

// Update stats
function updateStats() {
  const courses = loadCourses()
  const stats = [
    { 
      value: courses.length, 
      label: 'Активных курсов',
      icon: 'icons/stat-book.svg',
      color: '#6366F1'
    },
    { 
      value: courses.length > 0 ? Math.round(courses.reduce((sum, c) => sum + c.progress, 0) / courses.length) : 0, 
      label: 'Средний прогресс',
      icon: 'icons/stat-progress.svg',
      color: '#10B981'
    },
    { 
      value: courses.length > 0 ? (courses.reduce((sum, c) => sum + c.rating, 0) / courses.length).toFixed(1) : 0, 
      label: 'Средняя оценка',
      icon: 'icons/stat-rating.svg',
      color: '#F59E0B'
    }
  ]
  
  const statsCards = document.getElementById('statsCards')
  if (statsCards) {
    statsCards.innerHTML = stats.map(stat => `
      <div class="stat-card">
        <div class="stat-icon-wrapper" style="background: ${stat.color}15">
          <img src="${stat.icon}" alt="" width="32" height="32" />
        </div>
        <div class="stat-content">
          <div class="stat-value">${stat.value}${stat.label.includes('Прогресс') ? '%' : stat.label.includes('Оценка') ? '' : ''}</div>
          <div class="stat-label">${stat.label}</div>
        </div>
      </div>
    `).join('')
  }

  renderCourses()
}

// Render courses
function renderCourses() {
  const courses = loadCourses()
  const coursesGrid = document.getElementById('coursesGrid')
  
  if (!coursesGrid) return
  
  if (courses.length === 0) {
    coursesGrid.innerHTML = `
      <div class="empty-state">
        <img src="icons/logo-languages.svg" alt="" width="72" height="72" />
        <p>У вас пока нет активных курсов</p>
      </div>
    `
    return
  }
  
  const getLanguageColor = (language) => {
    const colors = {
      'Английский': '#3B82F6',
      'Испанский': '#EF4444',
      'Китайский': '#F59E0B',
      'Немецкий': '#10B981'
    }
    return colors[language] || '#6366F1'
  }
  
  coursesGrid.innerHTML = courses.map((course, index) => {
    const color = getLanguageColor(course.language)
    const stars = Array.from({ length: 5 }, (_, i) => 
      i < course.rating ? '★' : '☆'
    ).join('')
    
    return `
      <div class="course-card">
        <div class="course-header">
          <div class="course-icon-wrapper" style="background: ${color}15">
            <img src="${course.icon || 'icons/course-globe.svg'}" alt="${course.language}" width="40" height="40" />
          </div>
          <div class="course-info">
            <h3 class="course-language">${course.language}</h3>
            <p class="course-level">${course.level}</p>
          </div>
        </div>
        <div class="course-progress-section">
          <div class="progress-header">
            <span class="progress-label">Прогресс</span>
            <span class="progress-value">${course.progress}%</span>
          </div>
          <div class="progress-bar-wrapper">
            <div class="progress-bar" style="width: ${course.progress}%; background: ${color}"></div>
          </div>
        </div>
        <div class="course-rating">
          <span class="rating-label">Оценка:</span>
          <div class="stars-wrapper" style="color: #F59E0B; font-size: 1.2rem;">${stars}</div>
        </div>
      </div>
    `
  }).join('')
}

// Initialize enroll page
function initEnrollPage() {
  renderLanguages()
  
  const form = document.getElementById('enrollmentForm')
  const dateInput = document.getElementById('date')
  let selectedLanguage = ''
  
  // Set date limits
  const today = new Date()
  const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
  dateInput.min = today.toISOString().split('T')[0]
  dateInput.max = maxDate.toISOString().split('T')[0]
  
  // Language selection
  document.querySelectorAll('.language-card').forEach(card => {
    card.addEventListener('click', function() {
      document.querySelectorAll('.language-card').forEach(c => c.classList.remove('selected'))
      this.classList.add('selected')
      selectedLanguage = this.dataset.language
    })
  })
  
  // Form submit
  form.addEventListener('submit', function(e) {
    e.preventDefault()
    
    if (!selectedLanguage || !dateInput.value) {
      alert('Пожалуйста, выберите язык и дату старта')
      return
    }
    
    const savedCourses = localStorage.getItem('languageCourses')
    const courses = savedCourses ? JSON.parse(savedCourses) : []
    
    const languageData = languages.find(l => l.id === selectedLanguage)
    const newCourse = {
      id: Date.now(),
      language: languageData.name,
      level: 'Beginner',
      progress: 0,
      rating: 0,
      icon: languageData.icon
    }

    courses.push(newCourse)
    localStorage.setItem('languageCourses', JSON.stringify(courses))
    
    // Show success message
    const successMsg = document.getElementById('successMessage')
    successMsg.style.display = 'flex'
    
    setTimeout(() => {
      window.location.href = 'index.html'
    }, 2000)
  })
}

// Render languages
function renderLanguages() {
  const languagesGrid = document.getElementById('languagesGrid')
  if (!languagesGrid) return
  
  languagesGrid.innerHTML = languages.map(lang => `
    <div class="language-card" data-language="${lang.id}">
      <div class="language-icon-wrapper">
        <img src="${lang.icon}" alt="${lang.name}" width="40" height="40" />
      </div>
      <div class="language-content">
        <h3 class="language-name">${lang.name}</h3>
        <p class="language-description">${lang.description}</p>
      </div>
    </div>
  `).join('')
}

// Initialize slider
function initSlider() {
  const slider = document.getElementById('slider')
  if (!slider) return
  
  const slides = slider.querySelectorAll('.slide')
  const dots = slider.querySelectorAll('.dot')
  const prevBtn = document.getElementById('sliderPrev')
  const nextBtn = document.getElementById('sliderNext')
  
  let currentIndex = 0
  
  function showSlide(index) {
    slides.forEach((slide, i) => {
      slide.classList.toggle('active', i === index)
    })
    dots.forEach((dot, i) => {
      dot.classList.toggle('active', i === index)
    })
  }
  
  function nextSlide() {
    currentIndex = (currentIndex + 1) % slides.length
    showSlide(currentIndex)
  }
  
  function prevSlide() {
    currentIndex = (currentIndex - 1 + slides.length) % slides.length
    showSlide(currentIndex)
  }
  
  if (prevBtn) prevBtn.addEventListener('click', prevSlide)
  if (nextBtn) nextBtn.addEventListener('click', nextSlide)
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => {
      currentIndex = index
      showSlide(currentIndex)
    })
  })
  
  // Auto-advance every 3 seconds
  setInterval(nextSlide, 3000)
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', initPage)

